import { api } from "encore.dev/api";
import { SQLDatabase } from "encore.dev/storage/sqldb";

const db = SQLDatabase.named("madrasah");

interface Guru {
  id: string;
  nip: string;
  nama_lengkap: string;
  tempat_lahir: string;
  tanggal_lahir: string;
  jenis_kelamin: string;
  agama: string;
  alamat: string;
  no_telepon: string;
  email: string;
  pendidikan_terakhir: string;
  mata_pelajaran: string;
  status_kepegawaian: string;
  tanggal_masuk: string;
}

interface ListGuruResponse {
  guru: Guru[];
  total: number;
}

interface ListGuruParams {
  madrasah_id?: string;
  search?: string;
  limit?: number;
  offset?: number;
}

// Lists guru by madrasah
export const listGuru = api<ListGuruParams, ListGuruResponse>(
  { expose: true, method: "GET", path: "/guru" },
  async (params) => {
    const { madrasah_id, search = "", limit = 50, offset = 0 } = params;
    
    let whereClause = "";
    let searchParams: any[] = [];
    
    if (madrasah_id) {
      whereClause = "WHERE madrasah_id = $1";
      searchParams = [madrasah_id];
      
      if (search) {
        whereClause += " AND (nama_lengkap ILIKE $2 OR nip ILIKE $2 OR mata_pelajaran ILIKE $2)";
        searchParams.push(`%${search}%`);
      }
    } else if (search) {
      whereClause = "WHERE nama_lengkap ILIKE $1 OR nip ILIKE $1 OR mata_pelajaran ILIKE $1";
      searchParams = [`%${search}%`];
    }
    
    const guruList = await db.rawQueryAll<Guru>(
      `SELECT id, nip, nama_lengkap, tempat_lahir, tanggal_lahir, jenis_kelamin, agama, alamat, 
              no_telepon, email, pendidikan_terakhir, mata_pelajaran, status_kepegawaian, tanggal_masuk
       FROM guru ${whereClause} 
       ORDER BY nama_lengkap 
       LIMIT $${searchParams.length + 1} OFFSET $${searchParams.length + 2}`,
      ...searchParams, limit, offset
    );
    
    const totalResult = await db.rawQueryRow<{ count: number }>(
      `SELECT COUNT(*) as count FROM guru ${whereClause}`,
      ...searchParams
    );
    
    return {
      guru: guruList,
      total: totalResult?.count || 0
    };
  }
);
