import { api } from "encore.dev/api";
import { SQLDatabase } from "encore.dev/storage/sqldb";

const db = SQLDatabase.named("madrasah");

interface RecapData {
  madrasah_id?: string;
  nama_madrasah?: string;
  total_guru: number;
  total_siswa: number;
  total_siswa_laki: number;
  total_siswa_perempuan: number;
  total_siswa_kelas_1: number;
  total_siswa_kelas_2: number;
  total_siswa_kelas_3: number;
  total_siswa_kelas_4: number;
  total_siswa_kelas_5: number;
  total_siswa_kelas_6: number;
  total_karyawan: number;
  total_alumni: number;
}

interface RecapResponse {
  data: RecapData[];
  grand_total: {
    total_guru: number;
    total_siswa: number;
    total_siswa_laki: number;
    total_siswa_perempuan: number;
    total_siswa_kelas_1: number;
    total_siswa_kelas_2: number;
    total_siswa_kelas_3: number;
    total_siswa_kelas_4: number;
    total_siswa_kelas_5: number;
    total_siswa_kelas_6: number;
    total_karyawan: number;
    total_alumni: number;
  };
}

interface RecapParams {
  madrasah_id?: string;
}

export const getRecap = api<RecapParams, RecapResponse>(
  { expose: true, method: "GET", path: "/reports/recap" },
  async (params) => {
    const { madrasah_id } = params;
    
    if (madrasah_id) {
      const madrasah = await db.queryRow<{ nama_madrasah: string }>`
        SELECT nama_madrasah FROM madrasah WHERE id = ${madrasah_id}
      `;
      
      const guruCount = await db.queryRow<{ count: number }>`
        SELECT COUNT(*) as count FROM guru WHERE madrasah_id = ${madrasah_id}
      `;
      
      const siswaStats = await db.queryRow<{ 
        total: number;
        laki: number;
        perempuan: number;
        kelas_1: number;
        kelas_2: number;
        kelas_3: number;
        kelas_4: number;
        kelas_5: number;
        kelas_6: number;
      }>`
        SELECT 
          COUNT(*) as total,
          COUNT(*) FILTER (WHERE jenis_kelamin = 'Laki-laki') as laki,
          COUNT(*) FILTER (WHERE jenis_kelamin = 'Perempuan') as perempuan,
          COUNT(*) FILTER (WHERE kelas = '1') as kelas_1,
          COUNT(*) FILTER (WHERE kelas = '2') as kelas_2,
          COUNT(*) FILTER (WHERE kelas = '3') as kelas_3,
          COUNT(*) FILTER (WHERE kelas = '4') as kelas_4,
          COUNT(*) FILTER (WHERE kelas = '5') as kelas_5,
          COUNT(*) FILTER (WHERE kelas = '6') as kelas_6
        FROM siswa 
        WHERE madrasah_id = ${madrasah_id}
      `;
      
      const karyawanCount = await db.queryRow<{ count: number }>`
        SELECT COUNT(*) as count FROM karyawan WHERE madrasah_id = ${madrasah_id}
      `;
      
      const alumniCount = await db.queryRow<{ count: number }>`
        SELECT COUNT(*) as count FROM alumni WHERE madrasah_id = ${madrasah_id}
      `;
      
      const data: RecapData[] = [{
        madrasah_id,
        nama_madrasah: madrasah?.nama_madrasah || "",
        total_guru: guruCount?.count || 0,
        total_siswa: siswaStats?.total || 0,
        total_siswa_laki: siswaStats?.laki || 0,
        total_siswa_perempuan: siswaStats?.perempuan || 0,
        total_siswa_kelas_1: siswaStats?.kelas_1 || 0,
        total_siswa_kelas_2: siswaStats?.kelas_2 || 0,
        total_siswa_kelas_3: siswaStats?.kelas_3 || 0,
        total_siswa_kelas_4: siswaStats?.kelas_4 || 0,
        total_siswa_kelas_5: siswaStats?.kelas_5 || 0,
        total_siswa_kelas_6: siswaStats?.kelas_6 || 0,
        total_karyawan: karyawanCount?.count || 0,
        total_alumni: alumniCount?.count || 0,
      }];
      
      return {
        data,
        grand_total: {
          total_guru: guruCount?.count || 0,
          total_siswa: siswaStats?.total || 0,
          total_siswa_laki: siswaStats?.laki || 0,
          total_siswa_perempuan: siswaStats?.perempuan || 0,
          total_siswa_kelas_1: siswaStats?.kelas_1 || 0,
          total_siswa_kelas_2: siswaStats?.kelas_2 || 0,
          total_siswa_kelas_3: siswaStats?.kelas_3 || 0,
          total_siswa_kelas_4: siswaStats?.kelas_4 || 0,
          total_siswa_kelas_5: siswaStats?.kelas_5 || 0,
          total_siswa_kelas_6: siswaStats?.kelas_6 || 0,
          total_karyawan: karyawanCount?.count || 0,
          total_alumni: alumniCount?.count || 0,
        }
      };
    } else {
      const recapData = await db.queryAll<RecapData>`
        SELECT 
          m.id as madrasah_id,
          m.nama_madrasah,
          COALESCE(g.total_guru, 0) as total_guru,
          COALESCE(s.total_siswa, 0) as total_siswa,
          COALESCE(s.total_siswa_laki, 0) as total_siswa_laki,
          COALESCE(s.total_siswa_perempuan, 0) as total_siswa_perempuan,
          COALESCE(s.total_siswa_kelas_1, 0) as total_siswa_kelas_1,
          COALESCE(s.total_siswa_kelas_2, 0) as total_siswa_kelas_2,
          COALESCE(s.total_siswa_kelas_3, 0) as total_siswa_kelas_3,
          COALESCE(s.total_siswa_kelas_4, 0) as total_siswa_kelas_4,
          COALESCE(s.total_siswa_kelas_5, 0) as total_siswa_kelas_5,
          COALESCE(s.total_siswa_kelas_6, 0) as total_siswa_kelas_6,
          COALESCE(k.total_karyawan, 0) as total_karyawan,
          COALESCE(a.total_alumni, 0) as total_alumni
        FROM madrasah m
        LEFT JOIN (
          SELECT madrasah_id, COUNT(*) as total_guru 
          FROM guru 
          GROUP BY madrasah_id
        ) g ON m.id = g.madrasah_id
        LEFT JOIN (
          SELECT 
            madrasah_id,
            COUNT(*) as total_siswa,
            COUNT(*) FILTER (WHERE jenis_kelamin = 'Laki-laki') as total_siswa_laki,
            COUNT(*) FILTER (WHERE jenis_kelamin = 'Perempuan') as total_siswa_perempuan,
            COUNT(*) FILTER (WHERE kelas = '1') as total_siswa_kelas_1,
            COUNT(*) FILTER (WHERE kelas = '2') as total_siswa_kelas_2,
            COUNT(*) FILTER (WHERE kelas = '3') as total_siswa_kelas_3,
            COUNT(*) FILTER (WHERE kelas = '4') as total_siswa_kelas_4,
            COUNT(*) FILTER (WHERE kelas = '5') as total_siswa_kelas_5,
            COUNT(*) FILTER (WHERE kelas = '6') as total_siswa_kelas_6
          FROM siswa 
          GROUP BY madrasah_id
        ) s ON m.id = s.madrasah_id
        LEFT JOIN (
          SELECT madrasah_id, COUNT(*) as total_karyawan 
          FROM karyawan 
          GROUP BY madrasah_id
        ) k ON m.id = k.madrasah_id
        LEFT JOIN (
          SELECT madrasah_id, COUNT(*) as total_alumni 
          FROM alumni 
          GROUP BY madrasah_id
        ) a ON m.id = a.madrasah_id
        ORDER BY m.nama_madrasah
      `;
      
      const grandTotal = recapData.reduce(
        (acc, curr) => ({
          total_guru: acc.total_guru + curr.total_guru,
          total_siswa: acc.total_siswa + curr.total_siswa,
          total_siswa_laki: acc.total_siswa_laki + curr.total_siswa_laki,
          total_siswa_perempuan: acc.total_siswa_perempuan + curr.total_siswa_perempuan,
          total_siswa_kelas_1: acc.total_siswa_kelas_1 + curr.total_siswa_kelas_1,
          total_siswa_kelas_2: acc.total_siswa_kelas_2 + curr.total_siswa_kelas_2,
          total_siswa_kelas_3: acc.total_siswa_kelas_3 + curr.total_siswa_kelas_3,
          total_siswa_kelas_4: acc.total_siswa_kelas_4 + curr.total_siswa_kelas_4,
          total_siswa_kelas_5: acc.total_siswa_kelas_5 + curr.total_siswa_kelas_5,
          total_siswa_kelas_6: acc.total_siswa_kelas_6 + curr.total_siswa_kelas_6,
          total_karyawan: acc.total_karyawan + curr.total_karyawan,
          total_alumni: acc.total_alumni + curr.total_alumni,
        }),
        { 
          total_guru: 0, 
          total_siswa: 0,
          total_siswa_laki: 0,
          total_siswa_perempuan: 0,
          total_siswa_kelas_1: 0,
          total_siswa_kelas_2: 0,
          total_siswa_kelas_3: 0,
          total_siswa_kelas_4: 0,
          total_siswa_kelas_5: 0,
          total_siswa_kelas_6: 0,
          total_karyawan: 0, 
          total_alumni: 0 
        }
      );
      
      return {
        data: recapData,
        grand_total: grandTotal
      };
    }
  }
);
