import { ReactNode } from "react";
import { Link, useLocation, useNavigate } from "react-router-dom";
import { useAuth } from "../contexts/AuthContext";
import { Button } from "@/components/ui/button";
import { 
  School, 
  Users, 
  GraduationCap, 
  UserCheck, 
  Award, 
  LogOut, 
  Menu,
  LayoutDashboard
} from "lucide-react";
import { Sheet, SheetContent, SheetTrigger } from "@/components/ui/sheet";

interface LayoutProps {
  children: ReactNode;
}

export default function Layout({ children }: LayoutProps) {
  const { user, logout } = useAuth();
  const location = useLocation();
  const navigate = useNavigate();

  const handleLogout = () => {
    logout();
    navigate("/login");
  };

  const navigation = [
    { name: "Dashboard", href: "/dashboard", icon: LayoutDashboard },
    ...(user?.userType === "superadmin" 
      ? [{ name: "Madrasah", href: "/madrasah", icon: School }]
      : []
    ),
    { name: "Guru", href: "/guru", icon: Users },
    { name: "Siswa", href: "/siswa", icon: GraduationCap },
    { name: "Karyawan", href: "/karyawan", icon: UserCheck },
    { name: "Alumni", href: "/alumni", icon: Award },
  ];

  const NavigationItems = () => (
    <>
      {navigation.map((item) => {
        const Icon = item.icon;
        const isActive = location.pathname === item.href;
        return (
          <Link
            key={item.name}
            to={item.href}
            className={`flex items-center px-4 py-2 text-sm font-medium rounded-md transition-colors ${
              isActive
                ? "bg-teal-100 text-teal-700"
                : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
            }`}
          >
            <Icon className="mr-3 h-5 w-5" />
            {item.name}
          </Link>
        );
      })}
    </>
  );

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <header className="bg-white shadow-sm border-b">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center">
              <Sheet>
                <SheetTrigger asChild>
                  <Button variant="ghost" size="icon" className="md:hidden">
                    <Menu className="h-5 w-5" />
                  </Button>
                </SheetTrigger>
                <SheetContent side="left" className="w-64">
                  <div className="flex flex-col space-y-2 mt-6">
                    <NavigationItems />
                  </div>
                </SheetContent>
              </Sheet>
              
              <div className="flex items-center ml-4 md:ml-0">
                <div className="w-8 h-8 bg-green-600 rounded-full flex items-center justify-center relative mr-2">
                  <div className="text-white text-xs font-bold text-center leading-tight">
                    <div style={{ fontSize: '6px' }}>KKMI</div>
                    <div style={{ fontSize: '6px' }}>XII</div>
                  </div>
                  <div className="absolute -top-0.5 left-1/2 transform -translate-x-1/2">
                    <div className="w-1.5 h-1.5 bg-yellow-400 rounded-full"></div>
                  </div>
                </div>
                <span className="text-xl font-bold text-gray-900">ADAM</span>
              </div>
            </div>

            <div className="flex items-center space-x-4">
              <div className="text-sm text-gray-600">
                {user?.userType === "superadmin" ? (
                  <span className="font-medium">Super Admin</span>
                ) : user?.userType === "admin" ? (
                  <span className="font-medium">Admin - {user?.madrasahName}</span>
                ) : (
                  <span className="font-medium">Operator - {user?.madrasahName}</span>
                )}
              </div>
              <Button
                variant="ghost"
                size="sm"
                onClick={handleLogout}
                className="text-gray-600 hover:text-gray-900"
              >
                <LogOut className="h-4 w-4 mr-2" />
                Logout
              </Button>
            </div>
          </div>
        </div>
      </header>

      <div className="flex">
        {/* Sidebar - Desktop */}
        <nav className="hidden md:flex md:w-64 md:flex-col md:fixed md:inset-y-0 md:pt-16">
          <div className="flex-1 flex flex-col min-h-0 bg-white border-r">
            <div className="flex-1 flex flex-col pt-5 pb-4 overflow-y-auto">
              <div className="flex-1 px-3 space-y-2">
                <NavigationItems />
              </div>
            </div>
          </div>
        </nav>

        {/* Main content */}
        <main className="flex-1 md:ml-64">
          <div className="py-6">
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
              {children}
            </div>
          </div>
        </main>
      </div>
    </div>
  );
}
