import { useState } from "react";
import { useNavigate } from "react-router-dom";
import { useAuth } from "../contexts/AuthContext";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { useToast } from "@/components/ui/use-toast";
import { School } from "lucide-react";

export default function LoginPage() {
  const [username, setUsername] = useState("");
  const [password, setPassword] = useState("");
  const [isLoading, setIsLoading] = useState(false);
  const { login } = useAuth();
  const navigate = useNavigate();
  const { toast } = useToast();

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsLoading(true);

    try {
      await login(username, password);
      navigate("/dashboard");
      toast({
        title: "Login berhasil",
        description: "Selamat datang di ADAM",
      });
    } catch (error) {
      console.error("Login error:", error);
      toast({
        title: "Login gagal",
        description: "Username atau password salah",
        variant: "destructive",
      });
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-teal-50 to-teal-100 px-4">
      <Card className="w-full max-w-md">
        <CardHeader className="text-center">
          <div className="flex justify-center mb-4">
            <div className="p-3 bg-white rounded-full shadow-lg">
              <div className="w-16 h-16 bg-green-600 rounded-full flex items-center justify-center relative">
                <div className="absolute inset-0 rounded-full border-4 border-white"></div>
                <div className="text-white text-xs font-bold text-center leading-tight">
                  <div>KKMI</div>
                  <div>XII</div>
                </div>
                <div className="absolute -top-1 left-1/2 transform -translate-x-1/2">
                  <div className="w-3 h-3 bg-yellow-400 rounded-full"></div>
                </div>
              </div>
            </div>
          </div>
          <CardTitle className="text-2xl font-bold text-gray-900">
            ADAM
          </CardTitle>
          <CardDescription>
            Aplikasi Database Madrasah KKM 12 Suela-Sembalun
          </CardDescription>
        </CardHeader>
        <CardContent>
          <form onSubmit={handleSubmit} className="space-y-4">
            <div className="space-y-2">
              <Label htmlFor="username">Username</Label>
              <Input
                id="username"
                type="text"
                placeholder="NSM / NSM-OP / superadmin"
                value={username}
                onChange={(e) => setUsername(e.target.value)}
                required
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="password">Password</Label>
              <Input
                id="password"
                type="password"
                placeholder="Password"
                value={password}
                onChange={(e) => setPassword(e.target.value)}
                required
              />
            </div>
            <Button
              type="submit"
              className="w-full bg-teal-600 hover:bg-teal-700"
              disabled={isLoading}
            >
              {isLoading ? "Masuk..." : "Masuk"}
            </Button>
          </form>
          
          <div className="mt-4">
            <Button
              variant="outline"
              className="w-full border-2 border-teal-600 text-teal-600 hover:bg-teal-50"
              onClick={() => navigate("/register")}
            >
              Registrasi Operator Madrasah
            </Button>
          </div>
          
          <div className="mt-6 text-sm text-gray-600 space-y-2">
            <p className="font-medium">Petunjuk Login:</p>
            <ul className="space-y-1 text-xs">
              <li>• <strong>Super Admin:</strong> Username = superadmin, Password = admin123</li>
              <li>• <strong>Admin Madrasah:</strong> Username = NSM, Password = NPSN</li>
              <li>• <strong>Operator Madrasah:</strong> Username = NSM-OP, Password = Password yang didaftarkan</li>
            </ul>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
