import { api, APIError } from "encore.dev/api";
import { SQLDatabase } from "encore.dev/storage/sqldb";

const db = SQLDatabase.named("madrasah");

export interface RegisterOperatorRequest {
  nsm: string;
  token: string;
  password: string;
  confirm_password: string;
}

export interface RegisterOperatorResponse {
  message: string;
  madrasah_name: string;
}

export const registerOperator = api<RegisterOperatorRequest, RegisterOperatorResponse>(
  { expose: true, method: "POST", path: "/auth/register-operator" },
  async (req) => {
    const { nsm, token, password, confirm_password } = req;

    if (password !== confirm_password) {
      throw APIError.invalidArgument("Password dan konfirmasi password tidak sama");
    }

    if (password.length < 6) {
      throw APIError.invalidArgument("Password minimal 6 karakter");
    }

    const madrasah = await db.queryRow<{
      id: string;
      nama_madrasah: string;
      operator_token: string;
      operator_registered: boolean;
    }>`
      SELECT id, nama_madrasah, operator_token, operator_registered
      FROM madrasah 
      WHERE nsm = ${nsm}
    `;

    if (!madrasah) {
      throw APIError.notFound("NSM tidak ditemukan");
    }

    if (madrasah.operator_registered) {
      throw APIError.permissionDenied("Operator sudah terdaftar. Hubungi admin untuk reset token.");
    }

    if (madrasah.operator_token !== token) {
      throw APIError.unauthenticated("Token tidak valid");
    }

    const hashedPassword = Buffer.from(password).toString('base64');

    await db.exec`
      UPDATE madrasah 
      SET operator_password = ${hashedPassword},
          operator_registered = TRUE
      WHERE id = ${madrasah.id}
    `;

    return {
      message: "Registrasi berhasil! Silakan login dengan NSM-OP dan password Anda.",
      madrasah_name: madrasah.nama_madrasah
    };
  }
);
