import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { useAuth } from "../contexts/AuthContext";
import Layout from "../components/Layout";
import ImportDialog from "../components/ImportDialog";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { useToast } from "@/components/ui/use-toast";
import { Plus, Search } from "lucide-react";
import backend from "~backend/client";
import type { CreateAlumniRequest } from "~backend/alumni/create_alumni";

export default function AlumniPage() {
  const { user } = useAuth();
  const [search, setSearch] = useState("");
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [formData, setFormData] = useState<CreateAlumniRequest>({
    madrasah_id: user?.madrasahId || "",
    nisn: "",
    nis: "",
    nama_lengkap: "",
    tempat_lahir: "",
    tanggal_lahir: "",
    jenis_kelamin: "",
    agama: "",
    alamat: "",
    no_telepon: "",
    tahun_lulus: new Date().getFullYear(),
    melanjutkan_ke: "",
    pekerjaan: "",
    alamat_sekarang: "",
  });

  const { toast } = useToast();
  const queryClient = useQueryClient();

  const { data, isLoading } = useQuery({
    queryKey: ["alumni", user?.madrasahId, search],
    queryFn: () => backend.alumni.listAlumni({ 
      madrasah_id: user?.userType !== "superadmin" ? user?.madrasahId : undefined,
      search, 
      limit: 100 
    }),
  });

  const createMutation = useMutation({
    mutationFn: (data: CreateAlumniRequest) => backend.alumni.createAlumni(data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["alumni"] });
      setIsDialogOpen(false);
      setFormData({
        madrasah_id: user?.madrasahId || "",
        nisn: "",
        nis: "",
        nama_lengkap: "",
        tempat_lahir: "",
        tanggal_lahir: "",
        jenis_kelamin: "",
        agama: "",
        alamat: "",
        no_telepon: "",
        tahun_lulus: new Date().getFullYear(),
        melanjutkan_ke: "",
        pekerjaan: "",
        alamat_sekarang: "",
      });
      toast({
        title: "Berhasil",
        description: "Data alumni berhasil ditambahkan",
      });
    },
    onError: (error) => {
      console.error("Create alumni error:", error);
      toast({
        title: "Gagal",
        description: "Gagal menambahkan data alumni",
        variant: "destructive",
      });
    },
  });

  const importMutation = useMutation({
    mutationFn: (data: any[]) => {
      console.log("Import alumni data received:", data);
      
      const mappedData = data.map((row, index) => {
        console.log(`Processing alumni row ${index + 1}:`, row);
        
        return {
          nisn: String(row["NISN"] || row["nisn"] || "").trim(),
          nis: String(row["NIS"] || row["nis"] || "").trim(),
          nama_lengkap: String(row["Nama Lengkap"] || row["nama_lengkap"] || "").trim(),
          tempat_lahir: String(row["Tempat Lahir"] || row["tempat_lahir"] || "").trim(),
          tanggal_lahir: String(row["Tanggal Lahir"] || row["tanggal_lahir"] || "").trim(),
          jenis_kelamin: String(row["Jenis Kelamin"] || row["jenis_kelamin"] || "").trim(),
          agama: String(row["Agama"] || row["agama"] || "").trim(),
          alamat: String(row["Alamat"] || row["alamat"] || "").trim(),
          no_telepon: String(row["No Telepon"] || row["no_telepon"] || "").trim(),
          tahun_lulus: parseInt(String(row["Tahun Lulus"] || row["tahun_lulus"] || new Date().getFullYear())),
          melanjutkan_ke: String(row["Melanjutkan Ke"] || row["melanjutkan_ke"] || "").trim(),
          pekerjaan: String(row["Pekerjaan"] || row["pekerjaan"] || "").trim(),
          alamat_sekarang: String(row["Alamat Sekarang"] || row["alamat_sekarang"] || "").trim(),
        };
      });
      
      console.log("Mapped alumni data:", mappedData);
      return backend.alumni.importAlumni({
        madrasah_id: user?.madrasahId || "",
        data: mappedData
      });
    },
    onSuccess: (result) => {
      queryClient.invalidateQueries({ queryKey: ["alumni"] });
      toast({
        title: "Import Selesai",
        description: `Berhasil: ${result.success}, Gagal: ${result.failed}`,
      });
      if (result.errors.length > 0) {
        console.error("Import errors:", result.errors);
        toast({
          title: "Import Warnings",
          description: `Ada ${result.failed} data yang gagal diimport. Lihat console untuk detail.`,
          variant: "destructive",
        });
      }
    },
    onError: (error) => {
      console.error("Import alumni error:", error);
      toast({
        title: "Import Gagal",
        description: "Gagal mengimport data alumni",
        variant: "destructive",
      });
    },
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    createMutation.mutate(formData);
  };

  const handleImport = async (data: any[]) => {
    console.log("handleImport alumni called with:", data);
    await importMutation.mutateAsync(data);
  };

  const templateFields = [
    "NISN",
    "NIS",
    "Nama Lengkap",
    "Tempat Lahir",
    "Tanggal Lahir",
    "Jenis Kelamin",
    "Agama",
    "Alamat",
    "No Telepon",
    "Tahun Lulus",
    "Melanjutkan Ke",
    "Pekerjaan",
    "Alamat Sekarang"
  ];

  return (
    <Layout>
      <div className="space-y-6">
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">Data Alumni</h1>
            <p className="text-gray-600 mt-2">Kelola data alumni madrasah</p>
          </div>
          <div className="flex space-x-2">
            <ImportDialog
              title="Alumni"
              templateFields={templateFields}
              onImport={handleImport}
              isLoading={importMutation.isPending}
            />
            <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
              <DialogTrigger asChild>
                <Button className="bg-teal-600 hover:bg-teal-700">
                  <Plus className="h-4 w-4 mr-2" />
                  Tambah Alumni
                </Button>
              </DialogTrigger>
              <DialogContent className="max-w-2xl">
                <DialogHeader>
                  <DialogTitle>Tambah Alumni Baru</DialogTitle>
                </DialogHeader>
                <form onSubmit={handleSubmit} className="space-y-4">
                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="nisn">NISN</Label>
                      <Input
                        id="nisn"
                        value={formData.nisn}
                        onChange={(e) => setFormData({ ...formData, nisn: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="nis">NIS</Label>
                      <Input
                        id="nis"
                        value={formData.nis}
                        onChange={(e) => setFormData({ ...formData, nis: e.target.value })}
                      />
                    </div>
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="nama_lengkap">Nama Lengkap *</Label>
                    <Input
                      id="nama_lengkap"
                      value={formData.nama_lengkap}
                      onChange={(e) => setFormData({ ...formData, nama_lengkap: e.target.value })}
                      required
                    />
                  </div>
                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="tempat_lahir">Tempat Lahir</Label>
                      <Input
                        id="tempat_lahir"
                        value={formData.tempat_lahir}
                        onChange={(e) => setFormData({ ...formData, tempat_lahir: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="tanggal_lahir">Tanggal Lahir</Label>
                      <Input
                        id="tanggal_lahir"
                        type="date"
                        value={formData.tanggal_lahir}
                        onChange={(e) => setFormData({ ...formData, tanggal_lahir: e.target.value })}
                      />
                    </div>
                  </div>
                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="jenis_kelamin">Jenis Kelamin</Label>
                      <Input
                        id="jenis_kelamin"
                        value={formData.jenis_kelamin}
                        onChange={(e) => setFormData({ ...formData, jenis_kelamin: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="agama">Agama</Label>
                      <Input
                        id="agama"
                        value={formData.agama}
                        onChange={(e) => setFormData({ ...formData, agama: e.target.value })}
                      />
                    </div>
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="alamat">Alamat Saat Sekolah</Label>
                    <Input
                      id="alamat"
                      value={formData.alamat}
                      onChange={(e) => setFormData({ ...formData, alamat: e.target.value })}
                    />
                  </div>
                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="tahun_lulus">Tahun Lulus *</Label>
                      <Input
                        id="tahun_lulus"
                        type="number"
                        value={formData.tahun_lulus}
                        onChange={(e) => setFormData({ ...formData, tahun_lulus: parseInt(e.target.value) })}
                        required
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="no_telepon">No. Telepon</Label>
                      <Input
                        id="no_telepon"
                        value={formData.no_telepon}
                        onChange={(e) => setFormData({ ...formData, no_telepon: e.target.value })}
                      />
                    </div>
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="melanjutkan_ke">Melanjutkan Ke</Label>
                    <Input
                      id="melanjutkan_ke"
                      value={formData.melanjutkan_ke}
                      onChange={(e) => setFormData({ ...formData, melanjutkan_ke: e.target.value })}
                      placeholder="Nama sekolah/universitas lanjutan"
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="pekerjaan">Pekerjaan</Label>
                    <Input
                      id="pekerjaan"
                      value={formData.pekerjaan}
                      onChange={(e) => setFormData({ ...formData, pekerjaan: e.target.value })}
                      placeholder="Pekerjaan saat ini"
                    />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="alamat_sekarang">Alamat Sekarang</Label>
                    <Input
                      id="alamat_sekarang"
                      value={formData.alamat_sekarang}
                      onChange={(e) => setFormData({ ...formData, alamat_sekarang: e.target.value })}
                      placeholder="Alamat tempat tinggal saat ini"
                    />
                  </div>
                  <div className="flex justify-end space-x-2">
                    <Button
                      type="button"
                      variant="outline"
                      onClick={() => setIsDialogOpen(false)}
                    >
                      Batal
                    </Button>
                    <Button
                      type="submit"
                      className="bg-teal-600 hover:bg-teal-700"
                      disabled={createMutation.isPending}
                    >
                      {createMutation.isPending ? "Menyimpan..." : "Simpan"}
                    </Button>
                  </div>
                </form>
              </DialogContent>
            </Dialog>
          </div>
        </div>

        <Card>
          <CardHeader>
            <div className="flex items-center space-x-2">
              <Search className="h-4 w-4 text-gray-400" />
              <Input
                placeholder="Cari alumni..."
                value={search}
                onChange={(e) => setSearch(e.target.value)}
                className="max-w-sm"
              />
            </div>
          </CardHeader>
          <CardContent>
            {isLoading ? (
              <div className="text-center py-8">
                <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-teal-600 mx-auto"></div>
                <p className="text-gray-600 mt-2">Memuat data...</p>
              </div>
            ) : (
              <div className="overflow-x-auto">
                <table className="w-full">
                  <thead>
                    <tr className="border-b">
                      <th className="text-left py-3 px-4 font-medium text-gray-900">NISN</th>
                      <th className="text-left py-3 px-4 font-medium text-gray-900">Nama Lengkap</th>
                      <th className="text-left py-3 px-4 font-medium text-gray-900">Tahun Lulus</th>
                      <th className="text-left py-3 px-4 font-medium text-gray-900">Melanjutkan Ke</th>
                      <th className="text-left py-3 px-4 font-medium text-gray-900">Pekerjaan</th>
                      <th className="text-left py-3 px-4 font-medium text-gray-900">No. Telepon</th>
                    </tr>
                  </thead>
                  <tbody>
                    {data?.alumni.map((alumni) => (
                      <tr key={alumni.id} className="border-b hover:bg-gray-50">
                        <td className="py-3 px-4 text-sm">{alumni.nisn}</td>
                        <td className="py-3 px-4 text-sm font-medium">{alumni.nama_lengkap}</td>
                        <td className="py-3 px-4 text-sm">{alumni.tahun_lulus}</td>
                        <td className="py-3 px-4 text-sm">{alumni.melanjutkan_ke}</td>
                        <td className="py-3 px-4 text-sm">{alumni.pekerjaan}</td>
                        <td className="py-3 px-4 text-sm">{alumni.no_telepon}</td>
                      </tr>
                    ))}
                  </tbody>
                </table>
                {data?.alumni.length === 0 && (
                  <div className="text-center py-8 text-gray-600">
                    Tidak ada data alumni
                  </div>
                )}
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </Layout>
  );
}
